import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import java.io.*;

/**
 * A Simple text editor to demonstrate reading and writing from files.
 * This demonstrates one of the many ways to perform file I/O in Java.
 * 
 * A few comments in general before we get to the actual code:
 *
 * Generally when we do File I/O, we use two classes to give us access
 * to the file.  The first gives us access to the file, and the second
 * one gives us convenient methods to write to the file.  It looks
 * kind of like this:
 *    DataStream ---> FileStream --> File
 * We write to/read from the DataStream (or similar class), which
 * writes to the FileStream, which writes to the file.
 * This is done for a few reasons:
 * 1) It is usually more efficient to use things like DataStreams and
 *    PrintReaders/PrintWriters than to write directly to FileStreams
 *    and FileReaders.
 * 2) The DataStreams and PrintReaders/PrintWriters generally provide
 *    more convenient methods to perform file I/O.
 * Now onto the specifics of this code.
 
 * To begin with, we use a FileInputStream to read from the file.
 * This is used when you want to read a file as a binary file.
 * As we will see, a FileReader is used for text files.
 *
 */
public class TextEditor extends JFrame {

  JTextField fileNameInput= new JTextField();
  JTextArea theText= new JTextArea(6,40);

  int inputList[];
  int outputList[];
  
  public TextEditor() {
    Container contentPane =  this.getContentPane();
    contentPane.setLayout(new BorderLayout());
    this.setTitle("A very simple text editor");

    this.setSize(new Dimension(400,400));
    
    JLabel fileNameLabel=new JLabel("File Name: ");
    JButton readButton = new JButton("Read");
    JButton writeButton = new JButton("Write");
    
    Box buttonBox=Box.createHorizontalBox();
    buttonBox.add(fileNameLabel);
    buttonBox.add(fileNameInput);
    buttonBox.add(readButton);
    buttonBox.add(writeButton);
    
    this.getContentPane().add(buttonBox, BorderLayout.NORTH);
    this.getContentPane().add(theText, BorderLayout.CENTER);

    readButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        System.out.println("reading");
        readText();
      } });

    writeButton.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
        System.out.println("writing");
        writeText();
      } });
      
      pack();
      setVisible(true);
    }
    
  /**Overridden so we can exit when window is closed*/
  protected void processWindowEvent(WindowEvent e) {
    super.processWindowEvent(e);
    if (e.getID() == WindowEvent.WINDOW_CLOSING) {
      System.exit(0);
    }
  }
  /**
   * The main method so we can run the class as an application.
   */
  public static void main(String[] args) {
      new TextEditor();
  }

  void readText() {
      // File I/O can throw nasty exceptions, and they MUST be caught.
      // Thus, either the method has to throw the possible exceptions
      // that might be generated, or it has to deal with them itself.
      // In this case, we deal with it.  To do this, we use the regular
      // try-catch block.
      //
      try {
          String fileName=fileNameInput.getText();

          // Here we use a BufferedReader to read from a FileReader.
          // The FileReader gives us access to the file, but no nice
          // methods to read.  Also, file operations are expensive.
          // Thus we "wrap" the FileReader in a BuffereReader, which
          // will buffer data from the file so we are not reading 
          // each character from the file itself, which would be
          // really slow.
          // Most File I/O in Java will ivolve wrapping some sort
          // of File* class with some other class, often a Buffered*
          // class.
          // If you ever learn C/C++, you will be pleasantly suprised.
          // File I/O is one of the areas where C/C++ is much more 
          // straight forward and easy than Java (unlike inheritance,
          // polymorphism, etc.)
          //
          FileReader inStream=new FileReader(fileName);
          BufferedReader inData=new BufferedReader(inStream);

          // We read in the characters of the file one at a time,
          // appending them to the textArea.
          // Notice they are read as ints, not chars.
          // Thus, we cast them into chars as we append to textArea.
          theText.setText("");
          int in=inData.read();
          while(in!=-1) {
             theText.append(""+(char) in);
             in=inData.read();
          }
          inData.close();
     }      
      // Here is where we catch any exceptions that are thrown.
      // Notice we only catch an IOException.  This is the only
      // one we must catch in this case.
     catch(IOException e) {
          System.out.println("Error opening file");
          inputList=null;
     }
  }
  void writeText() {
      try {
          String fileName=fileNameInput.getText();

          // As with input, we wrap a FileWriter with another
          // writer.  This time we use a PrintWriter, since it
          // has methods to write all of the primitive types, and
          // Strings.  We use it to output the whole textArea as a 
          // String.
          //
          FileWriter outStream=new FileWriter(fileName);
          PrintWriter outData=new PrintWriter(outStream);

          String outputText=theText.getText();
          outData.print(outputText);
          outData.close();
          }
     catch(IOException e) {
          System.out.println("Error opening file");
          inputList=null;
     }
  }
}
