import javax.swing.event.EventListenerList;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.util.EventListener;
import java.util.Calendar;

/**
 * A Clock which keeps track of the current time.
 * It also has an alarm which can be set.
 * 
 * @author Charles Cusack 
 * @version August, 2008
 */
public class Clock extends Object
{
    // How long the alarm should go off, in minutes
    public static final int ALARM_LENGTH = 2; 
    
    private Time theTime;
    private Time alarmTime;
    private Time alarmEndTime;
    
    private boolean alarmSet;
    
    private javax.swing.Timer myTimer;
    
    // How often the clock should tick, in milliseconds.
    // Currently set to 60*1000 = one minute.
    // Setting it to 1000 will make it tick every second.
    // Setting it to 1 will make it tick every millisecond.
    private int tickLength = 60*1000;
    
    
    // Fields dealing with events.  These are used so that other
    // classes can listen to this class.  When this class changed,
    // it tells other classes so they can react if they wish.
    private EventListenerList listenerList = new EventListenerList();
    private ClockEvent clockEvent = null;
    
    /**
     * Constructor for objects of class Clock
     */
    public Clock()
    {
        // The following 4 lines get the current time
        Calendar now = Calendar.getInstance();
        int hours = now.get(Calendar.HOUR_OF_DAY);
        int minutes = now.get(Calendar.MINUTE);
        int seconds = now.get(Calendar.SECOND);
        
        theTime = new Time(hours,minutes);
        alarmTime = new Time();
        alarmEndTime = new Time();
        alarmSet=false;
        
        myTimer = new javax.swing.Timer(tickLength,new ActionListener() {        
            public void actionPerformed(ActionEvent e) {
                incrementTime();
            }
        });
        myTimer.start();
    }

    /**
     * Increment the time by one second.
     * Must check to see if the alarm should stop or start.
     */
    public void incrementTime() 
    {
        theTime.incrementMinutes();
        fireTimeChanged();
        if(theTime.equals(alarmTime)) 
        {
            fireAlarm();
        } 
        else if(theTime.equals(alarmEndTime))
        {
            fireEndAlarm();
        }
    }
    
    // You might think we should have one or more of the following
    // methods in this class.  However, this clock keeps real time,
    // so we do not allow the user to modify it.
    //
    // public void setTime(Time time) 
    // public void setTime(int hours, int minutes)
    
    
    /**
     * Set the alarm
     */
    public void setAlarmTime(Time alarmTime) 
    {
        this.alarmTime.setTime(alarmTime);
        this.alarmEndTime.setTime(alarmTime);
        this.alarmEndTime.addMinutes(ALARM_LENGTH);
    }
    public void setAlarm() {
        alarmSet = true;
    }
    public void unsetAlarm() {
        alarmSet = false;
    }
    
    public Time getTime()
    {
        return theTime;
    }    
    public Time getAlarmTime()
    {
        return alarmTime;
    }    
    
    //-----------------------------------------------------------
    // The methods below are all related to event handling  They 
    // allow other objects to request to be informed of changes
    // as well as allow them to actually be informed.
    
    /**
     * Add a listener to this class.  When a class is added as a 
     * listener, it will be informed of ClockEvents.
     * 
     * @param l the object which want to listen to this class.
     */
    public void addClockListener(ClockListener l) 
    {
        listenerList.add(ClockListener.class, l);
    }

    /**
     * Remove a listener from this class.  It will no longer recieve
     * notification of ClockEvents
     * 
     * @param l the object which no longer wants to listen to this class.
     */
    public void removeClockListener(ClockListener l) 
    {
        listenerList.remove(ClockListener.class, l);
    }
  
    /**
     * Inform all listeners that the time has changed.
     */
    protected void fireTimeChanged() 
    {
         // Guaranteed to return a non-null array
         Object[] listeners = listenerList.getListenerList();
         // Process the listeners last to first, notifying
         // those that are interested in this event
         for (int i = listeners.length-2; i>=0; i-=2) {
             if (listeners[i]==ClockListener.class) {
                 // Lazily create the event:
                 if (clockEvent == null)
                     clockEvent = new ClockEvent(this);
                 ((ClockListener)listeners[i+1]).timeChanged(clockEvent);
             }
         }
    }
   
    /**
     * Inform all listeners that the alarm has gone off.
     */
    protected void fireAlarm()    
    {
        if(!alarmSet)
        {
            return;
        }
         // Guaranteed to return a non-null array
         Object[] listeners = listenerList.getListenerList();
         // Process the listeners last to first, notifying
         // those that are interested in this event
         for (int i = listeners.length-2; i>=0; i-=2) {
             if (listeners[i]==ClockListener.class) {
                 // Lazily create the event:
                 if (clockEvent == null)
                     clockEvent = new ClockEvent(this);
                 ((ClockListener)listeners[i+1]).alarm(clockEvent);
             }
         }
    }
   
    /**
     * Inform all listeners that the alarm has stopped going off.
     */
    protected void fireEndAlarm()    
    {
         // Guaranteed to return a non-null array
         Object[] listeners = listenerList.getListenerList();
         // Process the listeners last to first, notifying
         // those that are interested in this event
         for (int i = listeners.length-2; i>=0; i-=2) {
             if (listeners[i]==ClockListener.class) {
                 // Lazily create the event:
                 if (clockEvent == null)
                     clockEvent = new ClockEvent(this);
                 ((ClockListener)listeners[i+1]).alarmEnded(clockEvent);
             }
         }
    }
}
