/**
 * A simple class to store time.
 * It stores time in Hours and Minutes.
 * 
 * This class is a modified version of ClockDisplay.
 * @author Michael Kolling and David J. Barnes
 * @version 2006.03.30
 * 
 * @author Modified by Charles Cusack, August, 2008.
 * 
 */
public class Time
{
    private NumberDisplay hours;
    private NumberDisplay minutes;
    
    /**
     * Constructor which sets time to 00:00
     */
    public Time()
    {
        hours = new NumberDisplay(24);
        minutes = new NumberDisplay(60);
    }

    /**
     * Constructor which allows you to set the time
     * to whatever you wish.
     * @param hour current hour
     * @param minute current minute
     */
    public Time(int hour, int minute)
    {
        hours = new NumberDisplay(24);
        minutes = new NumberDisplay(60);
        setTime(hour, minute);
    }

    /**
     * Returns the current minute
     * @return the current minute
     */
    public int getMinutes()
    {
        return minutes.getValue();
    }
    /**
     * Returns the current hour
     * @return the current hour
     */
    public int getHours()
    {
        return hours.getValue();
    }  
    
    /**
     * Sets minutes to minute
     * @param minute the number to set minute to
     */
    public void setMinutes(int minute)
    {
        minutes.setValue(minute);
    }  
    
    /**
     * Sets hours to hour
     * @param hour the number to set hour to
     */
    public void setHours(int hour)
    {
        hours.setValue(hour);
    }
    
    /**
     * Set the time of the display to the specified hour and minute
     * 
     * @param hour current hour
     * @param minute current minute
     */
    public void setTime(int hour, int minute)
    {
        
        setHours(hour);
        setMinutes(minute);
    }
   /**
     * Set the time of the display so it matches the time of
     * the argument.
     * @param time another Time object whose time you want this 
     *        Time object to be set to.
     */
    public void setTime(Time time)
    {
        setHours(time.getHours());
        setMinutes(time.getMinutes());
    }

   /**
     * Incremenent minutes by one.
     */
    public void incrementMinutes()
    {
        addMinutes(1);
    } 
   /**
     * Incremenent hours by one.
     */
    public void incrementHours()
    {
        addHours(1);
    } 
    
   /**
     * add min minutes to the minutes of this time.
     * If min > 60, add to hours.
     * 
     * Can you think of a way to improve this method?
     * (Hint:  You can get rid of the loop)
     * 
     * @param min number of minutes to add to minutes
     */
    public void addMinutes(int min)
    {
        while(min >= 60) {
            addHours(1);
            min -= 60;
        } 
        // If adding the minutes causes a rollover,
        // we need to increment the hours.
        if(minutes.getValue() + min >= 60)
        {
            minutes.add(min);
            addHours(1);
        }
        else
        {
            minutes.add(min);
        }   
    }
   /**
     * add hour hours to the hours of this time.
     * 
     * @param hour number of hours to add to hours
     */
    public void addHours(int hour)
    {
        hours.add(hour);
    }
  
    /**
     * Overriding the toString method so this time can be printed.
     * @return A String representation of this Time object, in the
     *         format "HH:MM"
     */
    public String toString()
    {
        return hours.getDisplayValue() 
        + ":" + minutes.getDisplayValue();
    }
    
    // The following methods are often useful to implement inclasses.
    // We will disucss these in the future.
    /**
     * @return a unique number for this time
     */
    public int hashCode() 
    {
        return 60*hours.getValue() + minutes.getValue();
    }
    /**
     * @param obj the object that is to be compared with this object
     * @return true if obj and this object are set to the same time, and
     *         false otherwise.
     */
    public boolean equals(Object obj)
    {
        if(this == obj)
        {
            return true;
        }
        if((obj == null) || (obj.getClass() != this.getClass()))
        {
            return false;
        }
        Time other = (Time) obj;
        if(hours.getValue() == other.hours.getValue() &&
           minutes.getValue() == other.minutes.getValue() )
        {
            return true;
        } 
        else 
        {
            return false;
        }  
    }
    
}
