/**
 * Shape.java
 * A class representing a drawable object that has a 
 * width, height, and indication of whether or not it
 * is filled in.
 * 
 * @author Chuck Cusack
 * @version January 2009
 */
import java.awt.Color;
import java.awt.Graphics;

public abstract class Shape extends DrawableObject {

	protected int width;
	protected int height;
	private boolean isFilled;

	/**
	 * In case a subclass doesn't know x, y, or color right away.
	 */
	protected Shape() {
		super(0,0,Color.black);
	}
	/**
	 * Constructor
	 * @param x the x-coordinate of the upper left corner
	 * @param y the y-coordinate of the upper left corner
	 * @param color the color of the object
	 */
	public Shape(int x, int y, Color color) {
		super(x, y, color);
		setWidth(0);
		setHeight(0);
		setFilled(false);
	}

	/**
	 * Constructor
	 * @param x the x-coordinate of the upper left corner
	 * @param y the y-coordinate of the upper left corner
	 * @param color the color of the object
	 * @param width the width of the shape
	 * @param height the height of the shape
	 * @param isFilled whether or not it is filled.
	 */
	public Shape(int x, int y, Color color, int width, int height, boolean isFilled) {
		super(x, y, color);
		this.width = width;
		this.height = height;
		this.isFilled = isFilled;
	}

	/**
	 * Set whether or not the Shape is filled.
	 * @param isFilled whether or not it is filled.
	 */
	public void setFilled(boolean isFilled) {
	   this.isFilled=isFilled;
	}

	/**
	 * @return isFilled
	 */
	public boolean getFilled() {
	   return isFilled;
	}
	
	/**
	 * @return width
	 */
	public int getWidth() {
	    return width;
	}

	
	/**
	 * Set the width of the shape
	 * @param width the width
	 */
	public void setWidth(int width) {
		   this.width=width;   
	}
	
	/**
	 * @return height
	 */
	public int getHeight() {
	    return height;
	}

	/**
	 * Set the height of the shape
	 * @param height the width
	 */
	public void setHeight(int height) {
	   this.height=height;   
	}

	/**
	 * Set the width and height of the shape
	 * @param width the width
	 * @param height the height
	 */
	public void setSize(int width, int height) {
	    setWidth(width);
	    setHeight(height);
	}

	/**
	 * Set the location and size of the Shape
	 * @param x the x location of the upper left corner
	 * @param y the y location of the upper left corner
	 * @param width the width
	 * @param height the height
	 */
	public void setLocationAndSize(int x, int y, int width,
			int height) {
			    setLocation(x,y);
			    setSize(width,height);
			}


	@Override
	public void drawObject(Graphics g) {
	    // Save the current color so we can reset it when we are done
	    Color oldColor=g.getColor();
	    
	    g.setColor(this.getColor());
	    if(isFilled) {
	        drawFilledShape(g);
	    } else {
	        drawHollowShape(g);   
	    }
	    // Reset the color back to the original.
	    g.setColor(oldColor);
	}
	
	/**
	 * Draw a highlight around the shape
	 */
	@Override
	public void drawHighlight(Graphics g) {
		 Color oldColor=g.getColor();
         g.setColor(Color.yellow);
         g.fillRect(getX()-2,getY()-2,getWidth()+4,getHeight()+4);  
         // Reset the color back to the original.
         g.setColor(oldColor);
	}
	/**
	 * The method to draw the shape if it is filled in.
	 */
	protected abstract void drawFilledShape(Graphics g);
	
	/**
	 * The method to draw the shape if it is not filled in.
	 */
	protected abstract void drawHollowShape(Graphics g);

}