import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/**
 * A class that stores a list of objects to draw on itself
 * 
 * @author Chuck Cusack
 * @version 1.0, September, 2005
 */
public class DrawingPanel extends JPanel implements KeyListener, MouseListener 
{
    // The list of GraphicalObjects on the panel
	ArrayList theObjects;
	
	// The item on the panel that is currently selected.
	GraphicalObject selectedObject;

	/**
	 * Constructor for objects of class DrawingPanel
	 */
	public DrawingPanel()
	{
		theObjects=new ArrayList();
		selectedObject=null;
		
		setFocusable(true);
        addMouseListener(this);
        addKeyListener(this);
        requestFocusInWindow();
		repaint();
	}
	
	/**
	 * Method overridden so that the JPanel canvas will be painted
	 * how we want it.
	 * For now, we just paint the various objects onto it.
	 */
	public void paintComponent(Graphics g) {
	    super.paintComponent(g);
	    for(int i=0;i<theObjects.size();i++) {
	           GraphicalObject go=(GraphicalObject) theObjects.get(i);
	           // Draw a yellow background for highlighted object.
	           if(go==selectedObject) {
	               Color oldColor=g.getColor();
	               g.setColor(Color.yellow);
	               int x=go.getX()-2;
	               int y=go.getY()-2;
	               int width,height;
	               if(go instanceof SizedObject) {
	                   width=((SizedObject) go).getWidth()+4;
	                   height=((SizedObject) go).getHeight()+4;
	               }
	               else {
	                   width=10;
	                   height=10;
	               }
	               g.fillRect(x,y,width,height);  
	               // Reset the color back to the original.
	               g.setColor(oldColor);
	           }
	           go.drawObject(g);
	    }
	}

    /**
     * Add the given object to the panel.
     * 
     * @param go The object to add to the panel.
     */
	public void addObject(GraphicalObject go) {
	        theObjects.add(go);
	        repaint();
	}
   /**
     * Remove the given object from the panel.
     * 
     * @param go The object to remove from the panel.
     * @returns true if the object was on the list and 
     *          it was removed, and false otherwise.
     */
	public boolean removeObject(GraphicalObject go) {
	        if(theObjects.remove(go)) {
	             repaint();
	             return true;
	        } else {
	             return false;
	        }
	}
	
	/** 
	 * @returns the GraphicalObject that is currently selected
	 */
	public GraphicalObject getSelectedObject() {
	    return selectedObject;
	}
	/**
	 * Sets the selected item to be the given object, if that
	 * object is on the list.  Otherwise, does nothing.
	 * @param object the item to make selected
	 * @returns true if object is on the list, and false otherwise
	 */
	public boolean setSelectedObject(GraphicalObject object) {
	    if(theObjects.contains(object)) {
	        selectedObject=object;
	        repaint();
	        return true;
	    } else {
	           return false;
	    }
	   }
   /**
	 * Sets the selected item to be the object with the
	 * given index, if the index is within the proper range.
	 * 
	 * @param index the index of the item to set as selected.
	 * @returns true if object is on the list, and false otherwise
	 */
	public boolean setSelectedObject(int index) {
	    if(index<0 || index >= theObjects.size()) {
	        return false;
	    } else {
	        selectedObject=(GraphicalObject) theObjects.get(index);
	        repaint();
	        return true;
	    }
	}
	
	/**
	 * @returns how many objects are on the panel
	 */
	public int getNumberObjects() {
	    return theObjects.size();
	}
	   
    /*
     * The following are from the KeyListener interface
     * 
     */
    public void keyPressed(KeyEvent e) {
        // This is implemented here and not in keyTyped because keyTyped
        // does not listen to non-printing keys (like arrow keys)
        // Very irritating, and this was tough to figure out, which is
        // why I am documenting it.
        if(selectedObject!=null) {
            switch (e.getKeyCode()) {
                case KeyEvent.VK_UP:
                    selectedObject.setY(selectedObject.getY()-10);
                    break;
                case KeyEvent.VK_DOWN:
                    selectedObject.setY(selectedObject.getY()+10);
                    break;
                case KeyEvent.VK_RIGHT:
                    selectedObject.setX(selectedObject.getX()+10);
                    break;
                case KeyEvent.VK_LEFT:
                    selectedObject.setX(selectedObject.getX()-10);
                    break;
                default:
            }
            repaint();
        }
    }
    public void keyTyped(KeyEvent e) {
        if(selectedObject!=null) {
            switch (e.getKeyChar()) { 
                case 't':
                    theObjects.remove(selectedObject);
                    theObjects.add(selectedObject);
                    break;
                case 'b':
                    theObjects.remove(selectedObject);
                    theObjects.add(0,selectedObject);
                    break;
                case 'n':
                    int index=theObjects.indexOf(selectedObject);
                    index=(index+1)%theObjects.size();
                    selectedObject=(GraphicalObject) theObjects.get(index);
                    break;
                default:
            }
            repaint();
        }
    }
    public void keyReleased(KeyEvent e) {
    }
    
        /*
     * Methods from the MouseListener interface
     * We don't need them all, but we have to implement them all
     * anyway.
     */
    public void mouseEntered(MouseEvent e) {
    }
    public void mouseExited(MouseEvent e) {
    }
    public void mouseClicked(MouseEvent e) { 
        // Need to do this so the panel gets focus so it can 
        // handle key events.  Without this, it doesn't get key events.
        requestFocusInWindow();
    }
    public void mousePressed(MouseEvent e) {
    }
    public void mouseReleased(MouseEvent e) {
    } 
}
